#!/usr/bin/env python3
# --------------------( LICENSE                           )--------------------
# Copyright (c) 2014-2021 Beartype authors.
# See "LICENSE" for further details.

'''
**Beartype callable utilities.**

This private submodule implements supplementary callable-specific utility
functions required by various :mod:`beartype` facilities, including callables
generated by the :func:`beartype.beartype` decorator.

This private submodule is *not* intended for importation by downstream callers.
'''

# ....................{ IMPORTS                           }....................
from beartype.cave import CallableTypes
from beartype.roar import _BeartypeUtilCallableException
from collections.abc import Callable
from sys import modules

# ....................{ GETTERS                           }....................
def get_callable_origin_label(func: Callable) -> str:
    '''
    Human-readable label describing the origin of the passed callable.

    Specifically, this getter returns either:

    * If this callable is pure-Python *and* physically declared on-disk, the
      absolute filename of the uncompiled on-disk Python script or module
      physically declaring this callable.
    * If this callable is pure-Python *and* dynamically declared in-memory,
      the placeholder string ``"<string>"``.
    * If this callable is C-based, the placeholder string ``"<C-based>"``.

    Caveats
    ----------
    **This getter is intentionally implemented for speed rather than robustness
    against unlikely edge cases.** The string returned by this getter is *only*
    intended to be embedded in human-readable labels, warnings, and exceptions.
    Avoid using this string for *any* mission-critical purpose.

    Parameters
    ----------
    func : Callable
        Callable to be inspected.

    Returns
    ----------
    str
        Either:

        * If this callable is physically declared by an uncompiled Python
          script or module, the absolute filename of this script or module.
        * Else, the placeholder string ``"<string>"`` implying this callable to
          have been dynamically declared in-memory.

    Raises
    ------
    _BeartypeUtilCallableException
        If this callable is *not* callable.

    See Also
    ----------
    :func:`inspect.getsourcefile`
        Inefficient stdlib function strongly inspiring this implementation,
        which has been highly optimized for use by the performance-sensitive
        :func:`beartype.beartype` decorator.
    '''

    # Avoid circular import dependencies.
    from beartype._util.func.utilfunccodeobj import get_func_codeobj_or_none

    # If this callable is *NOT*, raise an exception.
    if not callable(func):
        raise _BeartypeUtilCallableException(f'{repr(func)} not callable.')

    # Human-readable label describing the origin of the passed callable.
    func_origin_label = '<string>'

    # Else, this callable is a standard callable rather than an arbitrary class
    # or object defining the __call__() dunder method...
    if isinstance(func, CallableTypes):
        #FIXME: This is probably a bit overkill, as @beartype absolutely
        #*REQUIRES* pure-Python rather than C-based callables, as the latter
        #are *NOT* efficiently introspectable at runtime. *shrug*

        # Default this label to the placeholder string specific to C-based
        # callables to simplify subsequent logic.
        func_origin_label = '<C-based>'

        # Code object underlying this callable if found *OR* "None" otherwise.
        #
        # Note that we intentionally do *NOT* test whether this callable is
        # explicitly pure-Python or C-based: e.g.,
        #     # If this callable is implemented in C, this callable has no code
        #     # object with which to inspect the filename declaring this callable. In
        #     # this case, defer to a C-specific placeholder string.
        #     if isinstance(func, CallableCTypes):
        #         func_origin_label = '<C-based>'
        #     # Else, this callable is implemented in Python. In this case...
        #     else:
        #         # If this callable is a bound method wrapping an unbound function,
        #         # unwrap this method into the function it wraps. Why? Because only
        #         # the latter provides the code object for this callable.
        #         if isinstance(func, MethodBoundInstanceOrClassType):
        #             func = func.__func__
        #
        #         # Defer to the absolute filename of the Python file declaring this
        #         # callable, dynamically retrieved from this callable's code object.
        #         func_origin_label = func.__code__.co_filename
        #
        # Why? Because PyPy. The logic above succeeds for CPython but fails for
        # PyPy, because *ALL CALLABLES ARE C-BASED IN PYPY.* Adopting the above
        # approach would unconditionally return the C-specific placeholder
        # string for all callables -- including those originally declared as
        # pure-Python in a Python module. So it goes.
        func_codeobj = get_func_codeobj_or_none(func)

        # If this callable is has a code object, set this label to either the
        # absolute filename of the physical Python module declaring this
        # callable if this code object provides that metadata *OR* the current
        # placeholder string otherwise.
        #
        # Note that we intentionally do *NOT* assume all code objects to
        # provide this metadata (e.g., by unconditionally returning
        # "func_origin_label = func_codeobj.co_filename"). Why? Because PyPy
        # yet again. For inexplicable reasons, PyPy provides *ALL* C-based
        # builtins (e.g., len()) with code objects failing to provide this
        # metadata. Yes, this is awful. Yes, this is the Python ecosystem.
        if func_codeobj:
            func_origin_label = getattr(
                func_codeobj, 'co_filename', func_origin_label)
    # Else...
    else:
        # If this callable is *NOT* a class (i.e., is an object defining the
        # __call__() method), reduce this callable to the class of this object.
        if not isinstance(func, type):
            func = type(func)
        # In either case, this callable is now a class.

        # Fully-qualified name of the module declaring this class if this class
        # was physically declared by an on-disk module *OR* "None" otherwise.
        func_module_name = func.__module__

        # If this class was physically declared by an on-disk module, defer to
        # the absolute filename of that module.
        #
        # Note that arbitrary modules need *NOT* declare the "__file__" dunder
        # attribute. Unlike most other core Python objects, modules are simply
        # arbitrary objects that reside in the "sys.modules" dictionary.
        if func_module_name:
            func_origin_label = getattr(
                modules[func_module_name], '__file__', func_origin_label)

    # Return this func_origin_label.
    return func_origin_label
