import os
import sys
import pathlib
import subprocess


def run(args, **kwargs):
    print(f"running {args}")
    subprocess.run(args, check=True, shell=True, **kwargs)


netcfg_path = pathlib.Path(__file__).parent.joinpath("network.conf").resolve()
os.makedirs("build", exist_ok=True)
os.chdir("build")

boot_dir = "vexpress-boot"
boot_zip = boot_dir + ".zip"

linux_version = "5.10"
linux_folder = "linux-" + linux_version

print("--> make vexpress boot")
if os.path.isdir(boot_zip):
    print("nothing to do")
    sys.exit(0)

print("--> download linux")
if os.path.isdir(linux_folder):
    print("nothing to do")
else:
    run(
        [
            "git",
            "clone",
            "--depth=1",
            "--branch",
            f"v{linux_version}",
            "https://github.com/torvalds/linux/",
            linux_folder,
        ]
    )

os.chdir(linux_folder)

pathlib.Path(".config").unlink(missing_ok=True)

print("--> create dummy config from host")
run("make -j 4 ARCH=arm CROSS_COMPILE=arm-linux-gnueabihf- olddefconfig")

print("--> set linux configuration")
run("make ARCH=arm CROSS_COMPILE=arm-linux-gnueabihf- vexpress_defconfig")

with open(".config", "a") as config:

    # Disable HW_RANDOM
    config.write("\nCONFIG_HW_RANDOM=n\n")

    # Enable FUSE
    config.write("\nCONFIG_FUSE_FS=y\n")

    # Enable ExFAT
    config.write("\nCONFIG_EXFAT_FS=y\n")

    # Enable IPV6
    config.write("\nCONFIG_IPV6=y\n")

    # Enable NET_NS
    config.write("\nCONFIG_NET_NS=y\n")

    # Enable netfilter
    with open(netcfg_path, "r") as netcfg:
        config.write(netcfg.read())

print("--> compile linux")
run("make -j 4 ARCH=arm CROSS_COMPILE=arm-linux-gnueabihf- all")

print("--> create vexpress boot directory")
os.mkdir("../{}".format(boot_dir))
run(
    [
        "cp",
        ".config",
        "arch/arm/boot/zImage",
        "arch/arm/boot/dts/vexpress-v2p-ca15-tc1.dtb",
        f"../{boot_dir}",
    ]
)
run(
    [
        "cp",
        ".config",
        "arch/arm/boot/zImage",
        "arch/arm/boot/dts/vexpress-v2p-ca15_a7.dtb",
        f"../{boot_dir}",
    ]
)
run(
    [
        "cp",
        ".config",
        "arch/arm/boot/zImage",
        "arch/arm/boot/dts/vexpress-v2p-ca9.dtb",
        f"../{boot_dir}",
    ]
)
os.chdir("..")
with open("{}/README.txt".format(boot_dir), "w+") as readme:
    readme.write(
        "This is the kernel used by kiwix-hotspot to "
        "boot the vexpress machine in QEMU\n"
        "It has been generated by make-vexpress-boot script at "
        "https://github.com/kiwix/kiwix-hotspot"
    )
    readme.flush()

print("--> create vexpress boot zip archive")
run(f"zip -r --move {boot_zip} {boot_dir}")
